"use client";

import { useState } from "react";
import Image from "next/image";
import dynamic from "next/dynamic";

import type { ContactPageData, ContactFormRequest } from "@/src/types/contact";
import { sendContactForm } from "@/src/lib/sendContactForm";
import MapRoutingList from "@/app/components/map/MapRoutingList";

// Map just client
const ContactMap = dynamic(() => import("@/app/components/map/Map"), {
  ssr: false,
});

// HospitalDepartmentsSlider just client
const HospitalDepartmentsSlider = dynamic(
  () => import("@/app/components/common/HospitalDepartmentsSlider"),
  { ssr: false }
);

type Props = { data: ContactPageData };

export default function ContactPage({ data }: Props) {
  // ---------------- FORM STATE ----------------
  const [form, setForm] = useState<ContactFormRequest>({
    fullName: "",
    email: "",
    phone: "",
    subject: "",
    message: "",
  });

  const [loading, setLoading] = useState(false);
  const [successMsg, setSuccessMsg] = useState("");
  const [errorMsg, setErrorMsg] = useState("");

  const handleChange = (
    e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>
  ) => {
    setForm({ ...form, [e.target.name]: e.target.value });
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    setSuccessMsg("");
    setErrorMsg("");

    try {
      const res = await sendContactForm(form);

      if (res.success) {
        setSuccessMsg("پیام شما با موفقیت ارسال شد.");
        setForm({
          fullName: "",
          email: "",
          phone: "",
          subject: "",
          message: "",
        });
      } else {
        setErrorMsg(res.message || "خطا در ارسال پیام.");
      }
    } catch (err) {
      setErrorMsg("خطا در ارتباط با سرور. لطفاً دوباره تلاش کنید.");
    }

    setLoading(false);
  };

  return (
    <div className="contactPage">
      <div className="mainSmall">
        {/*  Top page */}
        <section className="contactPage_top flex flexSpace alignCenter whiteBox boxShadow radius20">
          {/* Contact Information */}
          <div className="contactInfoList radius16">
            <ul className="flex flexSpace alignStart">
              {data.infos.map((info) => (
                <li
                  key={info.id}
                  className={info.type === "address" ? "w100" : "w45"}
                >
                  <div className="contactInfoList_item">
                    <label className="fontBold">{info.label}</label>

                    {info.type === "address" ||
                    info.type === "time" ||
                    info.type === "fax" ? (
                      <p>{info.values[0]}</p>
                    ) : (
                      <div className="linkItem">
                        {info.values.map((v, i) =>
                          v.includes("@") ? (
                            <a key={i} href={`mailto:${v}`}>
                              {v}
                            </a>
                          ) : (
                            <a key={i} href={`tel:${v}`}>
                              {v}
                            </a>
                          )
                        )}
                      </div>
                    )}
                  </div>
                </li>
              ))}

              {/*Social media*/}
              <li className="w100">
                <div className="socialList">
                  <ul className="flex alignCenter">
                    {data.socials.map((s) => (
                      <li key={s.id}>
                        <a
                          href={s.url}
                          className="socialList_item radius50 flex flexCenter alignCenter"
                          target="_blank"
                          rel="noreferrer"
                        >
                          <Image src={s.icon} alt="" width={30} height={30} />
                        </a>
                      </li>
                    ))}
                  </ul>
                </div>
              </li>
            </ul>
          </div>

          {/*Contact form*/}
          <div className="contactForm">
            <div className="boldTitle">
              <strong>{data.title}</strong>
            </div>

            <div className="desc">
              <p>{data.description}</p>
            </div>

            <div className="publicForm w100">
              <form className="w100" onSubmit={handleSubmit}>
                <ul className="flex flexSpace">
                  <li className="relative w48">
                    <label className="absolute whiteBox">نام و نام خانوادگی</label>
                    <input
                      type="text"
                      name="fullName"
                      className="txt"
                      value={form.fullName}
                      onChange={handleChange}
                      required
                    />
                  </li>

                  <li className="relative w48">
                    <label className="absolute whiteBox">ایمیل</label>
                    <input
                      type="email"
                      name="email"
                      className="txt"
                      value={form.email}
                      onChange={handleChange}
                      required
                    />
                  </li>

                  <li className="relative w48">
                    <label className="absolute whiteBox">تلفن</label>
                    <input
                      type="tel"
                      name="phone"
                      className="txt"
                      value={form.phone}
                      onChange={handleChange}
                      required
                    />
                  </li>

                  <li className="relative w48">
                    <label className="absolute whiteBox">موضوع</label>
                    <input
                      type="text"
                      name="subject"
                      className="txt"
                      value={form.subject}
                      onChange={handleChange}
                      required
                    />
                  </li>

                  <li className="relative w100">
                    <label className="absolute whiteBox">توضیحات</label>
                    <textarea
                      name="message"
                      className="txtarea"
                      value={form.message}
                      onChange={handleChange}
                      required
                    ></textarea>
                  </li>

                  <li className="w100">
                    <input
                      type="submit"
                      className="addBtn bgGreen radius40 flex flexCenter alignCenter whiteTxt animationGreenBox w100"
                      value={loading ? "در حال ارسال..." : "ثبت"}
                      disabled={loading}
                    />
                  </li>
                </ul>

                {successMsg && (
                  <p className="successMessage greenTxt">{successMsg}</p>
                )}
                {errorMsg && (
                  <p className="errorMessage redTxt">{errorMsg}</p>
                )}
              </form>
            </div>
          </div>
        </section>

{/*  Map */}
<section className="contactPage_center">
  <div className="contactMap relative radius20">
    <ContactMap
      latitude={data.mapInfo.latitude}
      longitude={data.mapInfo.longitude}
    />

    <MapRoutingList 
      neshan={data.mapInfo.neshanUrlNavigation}
      google={data.mapInfo.googleMapUrlNavigation}
      waze={data.mapInfo.wazeUrlNavigation}
    />
  </div>
</section>



        {/*  Hospital Departmans */}
        <section className="contactPage_bottom">
          <div className="publicTitle">
            <div className="topTitle flex flexDirColumn alignCenter">
              <strong>
                <span>اطلاعات تماس بخش‌های </span>
                <label>بیمارستان فوق تخصص رضوی</label>
              </strong>
              <div className="desc">
                <p>
                  برای دریافت آدرس، شماره تلفن مستقیم و سایر راه‌های ارتباطی با
                  بخش‌های مختلف بیمارستان، از اطلاعات زیر استفاده نمایید.
                </p>
              </div>
            </div>
          </div>

          <HospitalDepartmentsSlider data={data.departments} />
        </section>
      </div>
    </div>
  );
}
