"use client";

import { useEffect, useState } from "react";
import { useRouter, useSearchParams } from "next/navigation";

type DropDownItem = {
  id: string;
  title: string;
};

export default function SearchBox() {
  const router = useRouter();
  const searchParams = useSearchParams();

  const [name, setName] = useState(searchParams.get("name") || "");
  const [specialty, setSpecialty] = useState(searchParams.get("expertiseId") || "");
  const [departmentId, setDepartmentId] = useState<string>("");

  const [departments, setDepartments] = useState<DropDownItem[]>([]);
  const [expertises, setExpertises] = useState<DropDownItem[]>([]);

  // API Internal — Departments
  useEffect(() => {
    async function loadDepartments() {
      try {
        const res = await fetch("/api/doctors/departments");
        const data = await res.json();
        setDepartments(data || []);
      } catch (err) {
        console.error("Error loading departments:", err);
      }
    }
    loadDepartments();
  }, []);

  // API Internal — Specialties
  useEffect(() => {
    async function loadExpertise() {
      try {
        const res = await fetch("/api/doctors/expertises");
        const data = await res.json();
        setExpertises(data || []);
      } catch (err) {
        console.error("Error loading expertises:", err);
      }
    }
    loadExpertise();
  }, []);

  // Send filters to URL
  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();

    const query = new URLSearchParams();

    if (name) query.set("name", name);
    if (specialty) query.set("expertiseId", specialty);
    if (departmentId) query.set("departmentId", departmentId);

    query.set("page", "1");

    router.push(`/doctors?${query.toString()}`);
  };

  return (
    <section className="drListPage_searchBox">
      <div className="publicForm w100">
        <form className="w100" onSubmit={handleSubmit}>
          <ul className="flex flexSpace">
            
            {/* Name */}
            <li className="relative">
              <label className="absolute whiteBox">نام و نام خانوادگی</label>
              <input
                type="text"
                className="txt"
                value={name}
                onChange={(e) => setName(e.target.value)}
              />
            </li>

            {/* Expertise */}
            <li className="relative">
              <label className="absolute whiteBox">تخصص</label>
              <select
                className="txt"
                value={specialty}
                onChange={(e) => setSpecialty(e.target.value)}
              >
                <option value="">همه</option>
                {expertises.map((exp) => (
                  <option key={exp.id} value={exp.id}>
                    {exp.title}
                  </option>
                ))}
              </select>
            </li>

            {/* department */}
            <li className="relative">
              <label className="absolute whiteBox">بخش</label>
              <select
                className="txt"
                value={departmentId}
                onChange={(e) => setDepartmentId(e.target.value)}
              >
                <option value="">همه</option>
                {departments.map((dep) => (
                  <option key={dep.id} value={dep.id}>
                    {dep.title}
                  </option>
                ))}
              </select>
            </li>

            {/* Search btn */}
            <li>
              <input
                type="submit"
                className="addBtn bgGreen radius40 flex flexCenter alignCenter whiteTxt animationGreenBox w100"
                value="جستجو"
              />
            </li>

          </ul>
        </form>
      </div>
    </section>
  );
}
