import Link from "next/link";
import { getHeaderMenu } from "@/src/lib/headerApi";
import { mockHeaderData } from "@/src/data/mockHeader";
import type { HeaderData } from "@/src/types/header";
import ScrollClassController from "@/app/components/header/ScrollClassController";
import HeaderClientWrapper from "@/app/components/header/HeaderClientWrapper";
import ResponsiveMenu from "@/app/components/header/ResponsiveMenu";


function normalizeMenu(items: any[]): HeaderData["mainMenu"] {
  return items.map((item) => ({
    id: item.id,
    title: item.title,
    url: item.url || "#",
    target: item.target || "_self",
    children: item.children?.length ? normalizeMenu(item.children) : undefined,
  }));
}

export default async function Header() {
  const menu = (await getHeaderMenu()) ?? [];

  const headerData: HeaderData = {
    ...mockHeaderData,
    mainMenu: menu.length ? normalizeMenu(menu) : [],
  };

  if (process.env.NODE_ENV === "development") {
    console.log(`🟢 Header Menu Loaded (${headerData.mainMenu.length} items)`);
  }

  return (
    <header id="mainNav" className="fix w100 transition">
      <HeaderClientWrapper />
      <ScrollClassController
        targetId="mainNav"
        scrolledClass="bg-white txt-black"
      />

      <div className="mainSmall radius30">

        <div className="headerUp flex flexSpace alignCenter">

          <div className="headerUp_r flex flexStart alignCenter">
            <div className="logoSec flex flexStart alignCenter">
              <div className="logo flex flexStart alignCenter">
                <Link href="/" className="bgWhite radius6 flex flexCenter alignCenter hoverImg">
                  <img src={headerData.logoMain} alt="لوگو" />
                </Link>
                <Link href="/" className="bgWhite radius6 flex flexCenter alignCenter hoverImg">
                  <img src={headerData.logoSecondary} alt="لوگو" />
                </Link>
              </div>
              <div className="headTitle whiteTxt flex flexDirColumn">
                <span className="txt14">{headerData.title}</span>
                <label className="txt12">{headerData.subtitle}</label>
              </div>
            </div>

            <div className="searchSec flex flexStart alignCenter radius40 whiteTxt">
              <div className="glassIcon flex flexCenter alignCenter">
                <i className="fa-solid fa-magnifying-glass"></i>
              </div>
              <input
                type="text"
                className="searchBox txt12"
                placeholder="جستجو بخش، کلینیک، پزشکان و ..."
              />
            </div>
          </div>

          <div className="headerUp_l flex flexEnd alignCenter">

            <div className="dropDown lang bgWhite radius40 animationWhiteBox relative">
              <div className="dropDown_active flex flexSpace alignCenter">
                <div className="flag">
                  <img
                    src="/images/IranFlag.png"
                    alt="فارسی"
                  />
                </div>
                <div className="flagTitle">
                  <span>فارسی</span>
                  <i className="hoverStyle transition fa-solid fa-angle-down"></i>
                </div>
              </div>
              <div className="langList bgWhite radius12 absolute w100">
                <ul>
                  {headerData.languages.map((lang, idx) => (
                    <li key={idx}>
                      <a href={lang.href} className="hoverStyle">
                        {lang.name}
                      </a>
                    </li>
                  ))}
                </ul>
              </div>
            </div>

            <div className="dropDown takingTurns greenBg radius40 animationGreenBox relative">
              <div className="takingTurns_item flex flexSpace alignCenter whiteTxt">
                <span>نوبت دهی و پذیرش</span>
                <i className="hoverStyle transition fa-solid fa-angle-down"></i>
              </div>
              <div className="takingTurns_subMenu absolute radius12 bgWhite w100">
                <ul>
                  {headerData.appointmentMenu.map((item, idx) => (
                    <li key={idx}>
                      <Link
                        href={item.href}
                        className="takingTurns_subMenu_item hoverStyle"
                      >
                        {item.title}
                      </Link>
                    </li>
                  ))}
                </ul>
              </div>
            </div>
          </div>
        </div>


        <div className="headerDown">
          <nav className="menu" role="navigation" aria-label="منوی اصلی">
            <ul className="flex flexSpace alignCenter">
              {headerData.mainMenu.map((item) => (
                <li
                  key={item.id}
                  className={`${
                    item.children
                      ? "submenuLi flex flexStart alignCenter hoverStyle submenuPro"
                      : ""
                  }`}
                >
                  <Link
                    href={item.url || "#"}
                    className="whiteTxt txt14 hoverStyle transition"
                  >
                    {item.title}
                  </Link>

                  {item.children && item.children.length > 0 && (
                    <>
                      <i className="fa-solid fa-angle-down whiteTxt txt14 hoverStyle transition"></i>
                      <div className="submenuPro-box">
                        <ul className="submenuPro-lvlOne">
                          {item.children.map((child) => (
                            <li key={child.id}>
                              <Link href={child.url || "#"}>{child.title}</Link>

                              {child.children && child.children.length > 0 && (
                                <ul className="submenuPro-lvlTwo">
                                  {child.children.map((sub) => (
                                    <li key={sub.id}>
                                      <Link href={sub.url || "#"}>
                                        {sub.title}
                                      </Link>
                                    </li>
                                  ))}
                                </ul>
                              )}
                            </li>
                          ))}
                        </ul>
                      </div>
                    </>
                  )}
                </li>
              ))}
            </ul>
          </nav>
        </div>
      </div>

      <ResponsiveMenu />
    </header>
  );
}
