"use client";

import { useState, useRef, useEffect } from "react";
import Image from "next/image";
import { Fancybox } from "@fancyapps/ui";
import "@fancyapps/ui/dist/fancybox/fancybox.css";
import type { HomeGallerySection as GallerySectionType } from "@/src/types/home";
import Link from "next/link";

type GallerySectionProps = {
  gallery: GallerySectionType;
};

export default function GallerySection({ gallery }: GallerySectionProps) {
  const [activeIndex, setActiveIndex] = useState(0);
  const [playingIndex, setPlayingIndex] = useState<number | null>(null);
  const videoRefs = useRef<(HTMLVideoElement | null)[]>([]);

  useEffect(() => {
    Fancybox.bind("[data-fancybox='gallery']", {});
    return () => {
      Fancybox.destroy();
    };
  }, []);

  const getVideoRef = (index: number) => (el: HTMLVideoElement | null) => {
    videoRefs.current[index] = el;
  };

  const togglePlay = (index: number) => {
    const video = videoRefs.current[index];
    if (!video) return;

    if (video.paused) {
      videoRefs.current.forEach((v, i) => {
        if (v && i !== index) {
          v.pause();
          v.currentTime = 0;
        }
      });
      video.play();
      setPlayingIndex(index);
    } else {
      video.pause();
      setPlayingIndex(null);
    }
  };

  const handleDoubleClick = (index: number) => {
    const video = videoRefs.current[index];
    if (video && video.requestFullscreen) {
      video.requestFullscreen();
    }
  };

  return (
    <section className="secFive">
      <div className="mainSmall">
       {/* Title */}
        <div className="publicTitle">
          <div className="topTitle flex flexDirColumn alignCenter">
            <strong>
              <span>{gallery.title}</span>
              <label>{gallery.subtitle}</label>
            </strong>
            <div className="desc">
              <p>{gallery.description}</p>
            </div>
          </div>
        </div>

        {/* Gallery */}
        <div className="imageGallery">
          <ul className="imageGallery_list flex flexSpace" id="galleryList">
            {gallery.items.map((item, idx) => (
              <li
                key={item.id}
                className={`radius40 overflow ${activeIndex === idx ? "active" : ""}`}
                onMouseEnter={() => setActiveIndex(idx)}
              >
                <div className="imageGallery_list_item relative w100 h100">
                  <figure className="relative w100 h100">
                    {item.type === "video" ? (
                      <>
                       <video
  ref={getVideoRef(idx)}
  className="w100 h100 objectfitCover cursor-pointer"
  poster={item.poster || undefined}
  onClick={() => togglePlay(idx)}
  onDoubleClick={() => handleDoubleClick(idx)}
>
  <source src={item.src} type="video/mp4" />
</video>

                        {/*  play icon */}
                        {playingIndex !== idx && (
                          <div
                            className="markIcon radius50 playBtn flex flexCenter alignCenter absolute cursor-pointer"
                            onClick={() => togglePlay(idx)}
                          >
                            <img src="/images/play-square.svg" alt="play" />
                          </div>
                        )}
                      </>
                    ) : (
                      <>
                        <a
                          href={item.src}
                          data-fancybox="gallery"
                          data-caption={item.caption || ""}
                          className="block w100 h100"
                        >
                          <Image
                            src={item.src}
                            alt={item.caption || "gallery image"}
                            width={400}
                            height={250}
                            className="w100 h100 objectfitCover"
                          />
                          {/* Image icon */}
                          <div className="markIcon radius50 flex flexCenter alignCenter absolute">
                            <img src="/images/imageIcon.svg" alt="image" />
                          </div>
                        </a>
                      </>
                    )}
                  </figure>

                  {item.caption && (
                    <figcaption className="imageGallery_list_txt absolute z-index w100">
                      <span>{item.caption}</span>
                    </figcaption>
                  )}
                </div>
              </li>
            ))}
          </ul>
        </div>

        {/* View All button */}
        <div className="allview">
          <Link
            href="/gallery"
            className="radius40 greenBg whiteTxt flex flexCenter alignCenter w100 h100 transition animationGreenBox"
          >
            مشاهده همه
          </Link>
        </div>
      </div>
    </section>
  );
}
