"use client";

import { useState, useEffect, useRef } from "react";
import Image from "next/image";
import type { HomeVirtualToursSection } from "@/src/types/home";
import { Fancybox } from "@fancyapps/ui";
import "@fancyapps/ui/dist/fancybox/fancybox.css";

type VirtualTourProps = {
  virtualTour: HomeVirtualToursSection;
};

export default function VirtualTour({ virtualTour }: VirtualTourProps) {
  const [activeIndex, setActiveIndex] = useState<number>(0);
  const [mounted, setMounted] = useState(false);
  const [playingIndex, setPlayingIndex] = useState<number | null>(null);

  const videoRefs = useRef<(HTMLVideoElement | null)[]>([]);

  const getVideoRef = (index: number) => (el: HTMLVideoElement | null) => {
    videoRefs.current[index] = el;
  };

  useEffect(() => {
    setMounted(true);
  }, []);

  const handlePlayVideo = (index: number) => {
    setPlayingIndex(index);

    videoRefs.current.forEach((video, idx) => {
      if (!video) return;

      if (idx === index) {
        video.play();
      } else {
        video.pause();
        video.currentTime = 0;
      }
    });
  };

  useEffect(() => {
    Fancybox.bind("[data-fancybox='gallery']", {});
    return () => Fancybox.destroy();
  }, []);

  return (
    <section className="secFive virtualTour">
      <div className="mainSmall">

       {/* Title */}
        <div className="publicTitle">
          <div className="topTitle flex flexDirColumn alignCenter">
              <strong>
              <span>{virtualTour.title}</span>
              <label>{virtualTour.subtitle}</label>
            </strong>

            <div className="desc">
              <p>{virtualTour.description}</p>
            </div>
          </div>
        </div>

        {/* Gallery */}
        <div className="imageGallery">
          <ul className="imageGallery_list flex flexSpace" id="virtualtour">
            {virtualTour.items.map((item, idx) => (
              <li
                key={idx}
                className={`radius40 overflow ${
                  mounted && activeIndex === idx ? "active" : ""
                }`}
                onMouseEnter={() => setActiveIndex(idx)}
              >
                <div className="relative w100 h100">
                  <figure className="imageGallery_list_img relative w100 h100">

                   {/* Video */}
                    {item.type === "video" ? (
                      <>
                        <video
                          ref={getVideoRef(idx)}
                          className="w100 h100 objectfitCover"
                          poster={item.poster ?? undefined}
                        >
                          <source src={item.src} type="video/mp4" />
                        </video>

                        {/* play icon فقط زمانی که ویدیو در حال پخش نیست */}
                        {playingIndex !== idx && (
                          <div
                            className="markIcon radius50 playBtn flex flexCenter alignCenter absolute z-index"
                            onClick={() => handlePlayVideo(idx)}
                          >
                            <img src="/assets/images/play-square.svg" alt="play" />
                          </div>
                        )}
                      </>
                    ) : (
                     /* Image */
                      <a
                        href={item.src}
                        data-fancybox="gallery"
                        className="block w100 h100"
                      >
                        <div className="markIcon radius50 flex flexCenter alignCenter absolute">
                          <img src="/assets/images/imageIcon.svg" alt="image" />
                        </div>

                        <Image
                          src={item.src}
                          alt={item?.caption ?? "gallery image"}
                          width={400}
                          height={250}
                          className="w100 h100 objectfitCover"
                        />
                      </a>
                    )}
                  </figure>

                  {item.caption && (
                    <figcaption className="imageGallery_list_txt absolute z-index w100">
                      <span>{item.caption}</span>
                    </figcaption>
                  )}
                </div>
              </li>
            ))}
          </ul>
        </div>
      </div>
    </section>
  );
}
