import type {
  FooterData,
  FooterLink,
  FooterLinksApiResponse,
  FooterInfoApiResponse,
} from "@/src/types/footer";

export async function getFooterData(): Promise<FooterData> {
  const API_BASE = process.env.NEXT_PUBLIC_API_BASE;

  const linksEndpoint = `${API_BASE}/Menu/api/v1/fa/footer-links`;
  const infoEndpoint = `${API_BASE}/General/api/v1/fa/footer-information`;

  try {
    // Fetch both APIs in parallel
    const [linksRes, infoRes] = await Promise.all([
      fetch(linksEndpoint, { cache: "no-store" }),
      fetch(infoEndpoint, { cache: "no-store" }),
    ]);

    if (!linksRes.ok) throw new Error(`Links HTTP ${linksRes.status}`);
    if (!infoRes.ok) throw new Error(`Info HTTP ${infoRes.status}`);

    const linksJson: FooterLinksApiResponse = await linksRes.json();
    const infoJson: FooterInfoApiResponse = await infoRes.json();

    const linkCategories = linksJson?.data?.links ?? [];

    const usefulCategory = linkCategories.find((c) =>
      c.title.includes("مفید")
    );
    const guideCategory = linkCategories.find((c) =>
      c.title.includes("مراجعین")
    );

    const usefulLinks: FooterLink[] = usefulCategory?.children ?? [];
    const visitorGuide: FooterLink[] = guideCategory?.children ?? [];

    const info = infoJson.data;

    return {
      footerText: info.footerText,
      contact: info.contact,
      socials: info.socialNetworks,
      badges: info.siteBadges,
      usefulLinks,
      visitorGuide,
    };
  } catch (err) {
    console.error("⚠️ Footer API error:", err);

    return {
      footerText: "",
      contact: {
        address: "",
        primaryPhone: "",
        fax: "",
        email: "",
        latitude: "",
        longitude: "",
        wazeUrlNavigation: "",
        googleMapUrlNavigation: "",
        neshanUrlNavigation: "",
      },
      socials: [],
      badges: [],
      usefulLinks: [],
      visitorGuide: [],
    };
  }
}
