import type { HomePageData } from "@/src/types/home";

function splitTitle(html: string): { title: string; subtitle: string } {
  if (!html) return { title: "", subtitle: "" };

  return {
    title: html.match(/<span>(.*?)<\/span>/)?.[1] ?? "",
    subtitle: html.match(/<label>(.*?)<\/label>/)?.[1] ?? "",
  };
}


function stripHtml(html: string): string {
  if (!html) return "";
  return html.replace(/<\/?[^>]+>/g, "").trim();
}

export async function getHomePage(): Promise<HomePageData> {
  const endpoint = `${process.env.NEXT_PUBLIC_API_BASE}/Pages/api/v1/fa/page/home`;

  const res = await fetch(endpoint, { cache: "no-store" });
  if (!res.ok) throw new Error("خطا در دریافت صفحه اصلی");

  const json = await res.json();
  const d = json.data;

  return {
    // ---------------------------------------------------------------------
    // Hero Slider
    // ---------------------------------------------------------------------
    bannersSection: {
      items: (d.bannersSection.items ?? []).map((item: any, i: number) => ({
        id: String(i),
        src: item.picture || null,
        title: item.title ?? "",
        description: item.description ?? "",
        url: item.url ?? "",
      })),
    },

    // ---------------------------------------------------------------------
    // Services Section
    // ---------------------------------------------------------------------
servicesSection: {
  ...splitTitle(d.servicesSection.title),
  description: d.servicesSection.description,
  items: (d.servicesSection.items ?? []).map((s: any) => ({
    title: s.title,
    src: s.picture || "",
    alt: s.title || "", 
    link: s.url || "#",
    description: s.description,
  })),
},


    // ---------------------------------------------------------------------
    // Doctors Section
    // ---------------------------------------------------------------------
    doctorsSection: {
      ...splitTitle(d.doctorsSection.title),
      description: d.doctorsSection.description ?? "",
      doctors: (d.doctorsSection.items ?? []).map((doc: any, idx: number) => ({
        id: String(idx),
        name: doc.fullName,
        specialty: doc.expertise,
        image: doc.picture,
        slug: doc.url,
      })),
    },

    // ---------------------------------------------------------------------
    // About Section 
    // ---------------------------------------------------------------------
    aboutSection: {
      title: d.aboutSection.title,
      subtitle: "", 
      description: stripHtml(d.aboutSection.description),

      link: "/about-us", 

      gallery: {
        big: d.aboutSection.centerPicture,
        small: d.aboutSection.rightPicture,
        smaller: d.aboutSection.leftPicture,
      },

      statistics: [
        {
          id: 1,
          value: Number(d.aboutSection.countServices ?? 0),
          label: "خدمات درمانی",
        },
        {
          id: 2,
          value: Number(d.aboutSection.countClinicalDepartments ?? 0),
          label: "بخش‌های بالینی",
        },
        {
          id: 3,
          value: Number(d.aboutSection.countHostPatients ?? 0),
          label: "بیماران پذیرش‌شده",
        },
        {
          id: 4,
          value: Number(d.aboutSection.countSpecialistMedicalStaff ?? 0),
          label: "پرسنل متخصص",
        },
      ],
    },

    // ---------------------------------------------------------------------
    // Virtual Tours
    // ---------------------------------------------------------------------
    virtualToursSection: {
      ...splitTitle(d.virtualToursSection.title),
  description: d.virtualToursSection.description,
  items: (d.virtualToursSection.items ?? []).map((v: any, i: number) => ({
    id: i + 1,
    type: v.type ?? "image",
    src: v.src ?? v.picture,
    poster: v.poster ?? null,
    caption: v.caption ?? null,
  })),
},


    // ---------------------------------------------------------------------
    // Gallery Section
    // ---------------------------------------------------------------------
   galleriesSection: {
  ...splitTitle(d.galleriesSection.title),
  description: d.galleriesSection.description,
  items: (d.galleriesSection.items ?? []).map((item: any, i: number) => ({
    id: i,
    type: "image",               
    src: item.picture,            
    poster: null,
    caption: item.title ?? "",
  })),
},


    // ---------------------------------------------------------------------
    // Comments
    // ---------------------------------------------------------------------
   commentsSection: {
  title: splitTitle(d.commentsSection.title).title,
  subtitle: splitTitle(d.commentsSection.title).subtitle,
  description: d.commentsSection.description,
  items: d.commentsSection.items.map((c: any) => ({
    fullName: c.fullName,
    comment: c.comment,
    picture: c.picture,
  })),
},

    // ---------------------------------------------------------------------
    // Insurances
    // ---------------------------------------------------------------------
    contractedInsurancesSection: {
  ...splitTitle(d.contractedInsurancesSection.title),
  description: d.contractedInsurancesSection.description,

  topRow: (d.contractedInsurancesSection.items ?? [])
    .slice(0, Math.ceil((d.contractedInsurancesSection.items ?? []).length / 2))
    .map((i: any) => i.picture),

  bottomRow: (d.contractedInsurancesSection.items ?? [])
    .slice(Math.ceil((d.contractedInsurancesSection.items ?? []).length / 2))
    .map((i: any) => i.picture),
},

    // ---------------------------------------------------------------------
    // Certificates
    // ---------------------------------------------------------------------
    certificatesSection: {
  ...splitTitle(d.certificatesSection.title),
  description: d.certificatesSection.description,
  certificates: (d.certificatesSection.items ?? []).map((i: any) => i.picture)
},


    // ---------------------------------------------------------------------
    // Blogs
    // ---------------------------------------------------------------------
blogsSection: (() => {
  const blogItems = Array.isArray(d.blogsSection.items)
    ? d.blogsSection.items
    : [];

  const top =
    blogItems.length > 0
      ? {
          title: blogItems[0].title ?? "",
          image: blogItems[0].image || null, 
          time: blogItems[0].time ?? "",
          link: blogItems[0].url ?? "#",
        }
      : {
          title: "",
          image: null,
          time: "",
          link: "#",
        };

  const newsList = blogItems.slice(1).map((i: any, idx: number) => ({
    id: idx + 1,
    title: i.title ?? "",
    image: i.image || null, 
    category: "خبر",
    time: i.time ?? "",
    link: i.url ?? "#",
  }));

  return {
    topNews: top,
    newsList,
  };
})(),

    // ---------------------------------------------------------------------
    // FAQs
    // ---------------------------------------------------------------------
   faqsSection: {
  ...splitTitle(d.faqsSection.title),
  items: (d.faqsSection.items ?? []).map((f: any, idx: number) => ({
    id: idx,
    question: f.question,
    answer: f.answer,
  })),
},

    // ---------------------------------------------------------------------
    // Social Networks
    // ---------------------------------------------------------------------
    socialNetworks: d.socialNetworks ?? [],
  };
}
